/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include "config_handler.h"
#include "numpad.h"
#include "statuscheck.h"
#include "gpioclassv2.h"
#include <QButtonGroup>
#include <QMainWindow>
#include <QMap>
#include <QPushButton>
#include <QTcpServer>
#include <QTcpSocket>
#include <QThread>
#include <QTimer>
#include <QtSerialPort/QSerialPort>
#include <QSerialPortInfo>
#include <QListWidgetItem>

#if defined (ROTARY_ENCODER_ENABLED)
#include "re_thread.h"
#endif

#define REMOTE_COMMAND_OFF			0
#define REMOTE_COMMAND_USB			1
#define REMOTE_COMMAND_TCP			2
#define REMOTE_COMMAND_RS232		3
#define REMOTE_COMMAND_RS485		4


namespace Ui {
class MainWindow;
}

class MainWindow : public QMainWindow
{
	Q_OBJECT

public:
	explicit MainWindow(QWidget *parent = 0);
	~MainWindow();

signals:
	void	buffer_fillup_detected();


/**********************************************************************************************************************************************************************************
 * SLOTS
 * *******************************************************************************************************************************************************************************/

private slots:
	//Miscellaneous Functions
	void	get_PowerInOut();		//get frequent power updates
	void	get_Temperature();
	void	get_IQmodVGA();
//	void	index_indicator();		//blinking index during power-set
	void	SG_error_handler(quint64 *error);
	void	SG_reset_handler();
	void	Suppress_RF_enable();
	void	DLL_frequency_indication();
	void	buffer_fillup_handler();

	//Raspberry Pi HAT B14_0835 Functions
#if defined(Q_OS_LINUX)
	void	LED_blink();
	void	LED_handler();
	void	handler_GPIO_PSU_button_enable();
	void	handler_GPIO_PSU_timer();
	bool	read_GPIO_PSU_button_enable();

#endif

	//Notification bar
	void	show_notification(QString message, int time = 8000);
	void	close_notification();
	void	show_warning(QString message);
	void	close_warning();

	//Miscellaneous buttons
	void	on_pushButton_RF_enable_1_clicked();
	void	on_pushButton_unit_power_reflected_1_clicked();
	void	on_pushButton_unit_power_forward_1_clicked();
	void	on_pushButton_unit_S11_1_clicked();
	void	on_pushButton_CW_enable_1_clicked();
	void	on_pushButton_PWM_enable_1_clicked();
	void	on_pushButton_PWM_enable_1_clicked(bool state);
	void	on_pushButton_ALL_enable_1_clicked();
	void	on_pushButton_DLL_enable_1_clicked();

	//Numpad Buttons
	void	on_numpadButton_0_clicked();
	void	on_numpadButton_1_clicked();
	void	on_numpadButton_2_clicked();
	void	on_numpadButton_3_clicked();
	void	on_numpadButton_4_clicked();
	void	on_numpadButton_5_clicked();
	void	on_numpadButton_6_clicked();
	void	on_numpadButton_7_clicked();
	void	on_numpadButton_8_clicked();
	void	on_numpadButton_9_clicked();
	void	on_numpadButton_sign_clicked();
	void	on_numpadButton_period_clicked();
	void	on_numpadButton_arrow_left_clicked();
	void	on_numpadButton_arrow_right_clicked();
	void	on_numpadButton_plus_clicked();
	void	on_numpadButton_minus_clicked();
	void	on_numpadButton_backspace_clicked();
	void	on_numpadButton_clear_all_clicked();
	void	on_numpadButton_ok_clicked();

	//Numpad Functions
	void	numpad_value_confirmed_handler(QPushButton *button, double *num);

	//Menu Buttons
	void	on_menu_home_Button_clicked();
	void	on_menu_ALL_Button_clicked();
	void	on_menu_DLL_Button_clicked();
//	void	on_menu_PID_Button_clicked();		//future?
	void	on_menu_sweep_Button_clicked();
	void	on_menu_settings_Button_clicked();
	void	on_menu_help_Button_clicked();
	void	on_menu_power_Button_clicked();
	void	on_menu_PSU_Button_clicked();

	//Home Menu
	void	on_pushButton_power_dbm_1_clicked();
	void	on_pushButton_power_watt_1_clicked();
	void	on_pushButton_frequency_1_clicked();
	void	on_pushButton_phase_1_clicked();
	void	on_pushButton_PWM_frequency_1_clicked();
	void	on_pushButton_PWM_duty_cycle_1_clicked();
	void	on_pushButton_ALL_threshold_1_clicked();
	void	on_pushButton_DLL_threshold_1_clicked();
	void	on_pushButton_DLL_step_frequency_1_clicked();

	void	on_pushButton_power_dbm_standalone_1_clicked();
	void	on_pushButton_power_SGx_dbm_1_clicked();
	void	on_pushButton_VGA_attenuation_1_clicked();
	void	on_pushButton_IQMod_magnitude_1_clicked();

	//ALL Menu
	void	on_pushButton_ALL_frequency_limit_upper_1_clicked();
	void	on_pushButton_ALL_frequency_limit_lower_1_clicked();
	void	on_pushButton_ALL_threshold_2_clicked();
	void	on_pushButton_DVGA_attenuation_forward_1_clicked();
	void	on_pushButton_DVGA_attenuation_reflected_1_clicked();
	void	on_pushButton_DVGA_amplifier_forward_1_clicked();
	void	on_pushButton_DVGA_amplifier_reflected_1_clicked();
	void	on_pushButton_ALL_reference_open_clicked();
	void	on_pushButton_ALL_reference_back_clicked();	//back button the reference page

	//DLL Menu
	void	on_pushButton_DLL_frequency_limit_upper_1_clicked();
	void	on_pushButton_DLL_frequency_limit_lower_1_clicked();
	void	on_pushButton_DLL_start_frequency_1_clicked();
	void	on_pushButton_DLL_step_frequency_2_clicked();
	void	on_pushButton_DLL_threshold_2_clicked();
	void	on_pushButton_DLL_main_delay_1_clicked();
	void	on_pushButton_power_dbm_2_clicked();
	void	on_pushButton_power_watt_2_clicked();

	//Sweep functions
	void	on_pushButton_SWP_start_frequency_1_clicked();
	void	on_pushButton_SWP_stop_frequency_1_clicked();
	void	on_pushButton_SWP_step_frequency_1_clicked();
	void	on_pushButton_SWP_power_dbm_1_clicked();
	void	on_pushButton_SWP_power_watt_1_clicked();
	void	on_pushButton_SWP_execute_1_clicked();
	void	on_pushButton_SWP_execute_2_clicked();
	void	on_pushButton_SWP_unit_clicked();
	void	on_pushButton_SWP_back_clicked();

	//PSU Menu
	void on_pushButton_PSU_voltage_1_clicked();
	void on_pushButton_PSU_current_limit_1_clicked();
	void on_pushButton_PSU_enable_1_clicked(bool checked);

	void handler_PSU_voltage_setpoint_get(double val);
	void handler_PSU_current_limit_get(double val);
	void handler_PSU_enable_combined_get(bool enable);
	void handler_PSU_IU_get(int psu_num, double voltage, double current, double power);
	void handler_PSU_power_efficiency_get(double efficiency);
	void handler_PSU_dissipation_get( double dissipation);

	//Settings Menu
	void	on_pushButton_power_control_normal_1_clicked();	//Autogain Enabled
	void	on_pushButton_power_control_analog_1_clicked();	//Analog Input Mode
	void	on_pushButton_power_control_feedforward_1_clicked();	//Feed Forward Power Control
	void	on_pushButton_power_control_amplifier_1_clicked();	//Amplifier Mode
	void	on_pushButton_clock_source_standalone_1_clicked();				//(Daisy) Chain Off
	void	on_pushButton_clock_source_master_1_clicked();				//(Daisy) Chain Master
	void	on_pushButton_clock_source_slave_inline_1_clicked();				//(Daisy) Chain Slave
	void	on_pushButton_clock_source_slave_end_1_clicked();				//End of chain slave //not used
	void	on_pushButton_PWM_triggering_free_running_1_clicked();		//Free-running
	void	on_pushButton_PWM_triggering_master_1_clicked();		//Triggered Master
	void	on_pushButton_PWM_triggering_slave_1_clicked();		//Triggered Slave
	void	on_pushButton_remote_command_OFF_1_clicked();					//RCM Off
	void	on_pushButton_remote_command_USB_1_clicked();					//RCM USB
	void	on_pushButton_remote_command_RS232_1_clicked();					//RCM UART -> RS232 via RPi HAT
	void	on_pushButton_remote_command_RS485_1_clicked();					//RCM UART -> RS485 via RPi HAT
	void	on_pushButton_remote_command_TCP_1_clicked();					//RCM TCP
	void	on_pushButton_external_triggering_OFF_1_clicked();	//External Triggering OFF
	void	on_pushButton_external_triggering_ON_1_clicked();	//External Triggering ON

	//Help Menu
	void	on_listWidget_currentItemChanged(QListWidgetItem *current, QListWidgetItem *previous);


	//Remote Command mode
	void	UART_RCM_Pass();
	void	UART_RCM_Return();
	void	UART_disconnected();
	void	TCP_RCM_Pass();
	void	TCP_RCM_Return();
	void	TCP_newConnection();
	void	TCP_disconnect_server();
	void	TCP_client_disconnected();
	void	RCM_controls_enable(bool state);

	//Power menu
	void	on_pushButton_shutdown_controller_1_clicked();
	void	on_pushButton_reboot_controller_1_clicked();
	void	on_pushButton_exit_program_1_clicked();
	void	on_Exit_Yes_button_clicked();
	void	on_Exit_No_button_clicked();

#ifdef ROTARY_ENCODER_ENABLED
	//Rotary Encoder Handlers
	void	thread_ErrorHandler1(QString);
	void	thread_Knob_enterHandler();
	void	thread_Knob_leftHandler();
	void	thread_Knob_rightHandler();
	void	thread_ManualHandler();
	void	thread_TransparentHandler();
	void	thread_AccelerationHandler();
	void	thread_DecelerationHandler();
	void	RE_update_values();
	void	on_RF_enableButton_2_clicked();
#endif

/**********************************************************************************************************************************************************************************
 * Variables and functions
 * *******************************************************************************************************************************************************************************/

private:
	Ui::MainWindow *ui;
	ConfigHandler *config;

	/* GPIOs for Raspberry Pi */
#if defined(Q_OS_LINUX)
	GPIOClass *GPIO_control_external;
	GPIOClass *GPIO_control_SGx;
	GPIOClass *GPIO_HAT_serial_mode;
	GPIOClass *GPIO_RS485_RW_flipper;   //0 = read, 1 = transmit
	GPIOClass *GPIO_LED_1;
	GPIOClass *GPIO_interlock_out;
	GPIOClass *GPIO_PSU_button_enable;
	QTimer	*GPIO_PSU_timer;
	bool last_GPIO_PSU_button_enable_state = false;
	int last_LED_mode = -1;

#endif

	/* Important defaults */
	int		PA_type = 0;
	void	Power_control_mode_handler();
	int		power_control_mode;

	/* Analog Input Mode */
	void	Analog_input_mode_enable(int state, int mode);

	int		AIS_modulation_source;
	int		AIS_attenuation_value;

	/* External triggering mode */
	void	External_triggering_enable(bool state);
	void	External_triggering_handler();
	int		ext_trig_mode;

	/* Transparent Mode */
	void	RCM_mode_handler();
	int		remote_command_mode = REMOTE_COMMAND_OFF;		//0 = OFF, 1 = UART/USB, 2 = TCP
	QString ipAddress;						//IP Address of this system

	/* Coherency mode */
	void	Coherency_mode_handler();
	bool	coherency_supported = false;
	int		clock_source = 0;				//0 = standalone mode, 1 = master, 3 = Slave, 2 = (not used) End of chain Slave

	/* Trigger mode */
	void	PWM_triggering_mode_handler();
	int		PWM_trigger_mode;				//1 = Freerunning, 2 = Triggered master, 3 = Triggered Slave

	/* CW variables */
	double	power_watt_value = 0.1;			//Starting value Power	(Watt)
	double	power_dbm_value = 20.0;			//Starting value Power (dBm) <- Not assigned from config
	double	freq_value = 2450000000.0;		//Starting value Frequency
	double	phase_value = 0.0;				//Starting value Phase

	//
	// TODO:
	// Not very nice to have standalone power limits hardcoded, but it'll do for now.
	//
	double	power_dbm_standalone_value = 0;		//PWRDS in for standalone PA type.
	double	power_dbm_standalone_max = 30;
	double	power_dbm_standalone_min = -30;
	double	sgx_power_value = 0;
	double	VGA_attenuation_value = 0;
	double	magnitude_value = 0;

	/* PWM variables */
	bool	PWM_enabled = false;
	//bool	PWM_capable = true;
	double 	PWM_freq_value = 1000;
	double	PWM_duty_cycle_value = 100.0;
	double  PWM_duty_cycle_min = 1.0;
//	double  PWM_duty_cycle_max = 99.0;
	double	PWM_correction_factor = 0.0;	//TODO: idk what this value is for...
	double	PWM_delay = 0.0;				//TODO: idk what this value is for...
	int		PWM_master_port = 0;		//Purpose Unknown
	int		PWM_master_pin = 0;			//Purpose Unknown
	int		PWM_slave_port = 0;			//Purpose Unknown
	int		PWM_slave_pin = 0;			//Purpose Unknown

	/* ALL variables */
	bool	ALL_enabled = false;			//check whether it actually is or isn't enabled
	double	ALL_freqLimit_UP_value = 1.2;
	double	ALL_freqLimit_DOWN_value = 0.85;
	double	ALL_threshold_value = 0.2;
	double	DVGA_attenuation_forward = 9.0;
	double	DVGA_attenuation_reflected = 9.0;
	bool	DVGA_amplifier_forward_enabled = false;
	bool	DVGA_amplifier_reflected_enabled = false;

	/* DLL variables */
	bool	DLL_enabled = false;
	double	DLL_freqLimit_DOWN_value = 2400.0;
	double	DLL_freqLimit_UP_value = 2500.0;
	double  DLL_freq_start_value = 2400.0;
	double  DLL_freq_step_value = 1.0;
	double	DLL_threshold_value = 0.5;
	double  DLL_main_delay_value = 0;
	double	DLL_main_delay_min = 0;
	double  DLL_main_delay_max = 1000;
	double  DLL_threshold_max = 60;
	double  DLL_settle_frequency = 2433.0;	//used for frequency indication during DLL
	//bool	DLL_capable = true;

	/* SWEEP variables */
	int		SWP_run_sweep();
	void	SWP_draw_plot();
	double	SWP_start_freq = 2400000000.0;
	double	SWP_stop_freq = 2500000000.0;
	double	SWP_step_freq = 5000000.0;
	double	SWP_maximum_measure_points = 51;
	double	SWP_power_watt = 0.001;				//relies on dbm value above		//cannot be 0!
	double	SWP_power_dbm = 0.0;
	QVector<double>	SWP_freq_data;
	QVector<double>	SWP_frw_data;
	QVector<double>	SWP_rfl_data;
	QVector<double>	SWP_s11_dbm_data;
	QVector<double>	SWP_s11_watt_data;
	QStringList	SWP_data;
	QString		SWP_raw_data;
	QString		SWP_mode = "dbm";

	/* PSU variables */
	void handler_PSU_reset_to_default();		//deals with PSU configuration according to the config file
	void get_PSU_IU();
	int PSU_count = 0;
	double PSU_voltage_setpoint = 0;
	double PSU_current_limit = 0;
	bool PSU_enable_combined = 0;
	//double PSU_voltage_combined = 0;
	//double PSU_current_combined = 0;
	QList<bool> PSU_enable;
	QList<double> PSU_voltage;
	QList<double> PSU_current;
	QList<double> PSU_power;
	double PSU_total_power;
	double PSU_power_efficiency;
	double PSU_dissipation_watt;

	/* CORE */
	QSerialPortInfo	*info;
	QSerialPort	SG_port;
	QSerialPort	RCM_port;
	bool RCM_supported = false;
	QTcpServer	*tcpServer = nullptr;
	QTcpSocket	*tcpSocket = nullptr;
	QString		channel_select = "1";
	StatusCheck *status_checker;

	QTimer	*readings_timer;
	QTimer	*notification_timer;

#if defined(Q_OS_LINUX)
	QTimer  *LED_timer;
#endif

	double	PowerFRW_Value = 0.0;
	double	PowerRFL_Value = 0.0;
	double	Temperature_Value = 0.0;
	double	Temperature_average = 0.0;
	QVector<double> Temperature_array;
	bool	RF_enabled = false;			//Actual RF power / ECS

	/* Startup functions */
	QList<QSerialPortInfo>	get_SGX_port();
	QList<QSerialPortInfo>	get_RCM_port();
	bool	port_open(QSerialPort *port);
	void	handle_SG_type();
	void	startup_sequence();
	void	sync_to_config();
	void	run_all_gets();
	void	update_labels();
	void	update_buttons();
	void	handle_config_targets();
	void	startup_check_minmax();
	void	safe_reset();
	void	reset_restore();
	void	version_control();

	QListWidgetItem listWidgetItem_system;
	QListWidgetItem listWidgetItem_about;
	QListWidgetItem listWidgetItem_license;
	QListWidgetItem listWidgetItem_copyright;

	QString	SG_firmware_version;
	QString	toolkit_GUI_version;
	std::array<int,4> firmware_version_actual;
	std::array<int,4> firmware_version_requirement;
	bool	bootup = true;			//Necessary for handling resets differently before and after program startup / prevents an infinite loop related to RCM

	/* Miscellaneous */
	void	show_DLL_settings(bool input);
	void	show_ALL_settings(bool input);
	void	show_PWM_settings(bool input);
	void	show_autogain_controls(bool enable);
	void	show_Amplifier_mode_controls(bool enable);
	void	show_power_dbm_standalone_controls(bool enable);
	void	show_SGx_Power_controls(bool enable);
	void	show_GCS_controls(bool enable);
	void	show_MCS_controls(bool enable);
	void	change_power_reading_units(QPushButton *state);
	void	menuButton_manager(QPushButton *menuButton_pointer);	//controls behaviour of menu buttons / page hopping

	QButtonGroup menu_buttonGroup;
	QButtonGroup chain_mode_buttonGroup;
	QButtonGroup RCM_buttonGroup;
	QButtonGroup PWM_triggering_buttonGroup;
	QButtonGroup Power_control_mode_buttonGroup;
	QButtonGroup External_triggering_buttonGroup;
	QButtonGroup controls_buttonGroup;
	QString		SG_manufacturer = "-";		//Manufacturer of SG Board
	QString		SG_board_model = "-";		//Type of SG Board
	QString		SG_serial_number = "-";		//Serial number of the SG Board
	quint64		SG_status;				//SG error status
	int			last_page_index;		//Track the last menu visited for some miscellaneous use-cases
	int			power_menu_action;		//Choose Shutdown / Restart / Exit

	/* Operational mode functions */
	void	ALL_mode_enable(bool state);	//enable/disable switch
	void	DLL_mode_enable(bool state);
	void	PWM_mode_enable(bool state);
	void	DVGA_enable();					//function which handles mode swapping and deals with the DCA values appropriately

	/* Parser functions */
	void	show_warning_popup(QString status);
	bool	serial_parser(QString status);			//Checks if a set function returns OK

	//Get parsers, read and process the results of Get commands.
	void	ALL_ALCG_parser();	//ALL config
	void	ALL_ALEG_parser();	//ALL enable
	void	DLL_DLCG_parser();	//DLL config
	void	DLL_DLEG_parser();	//DLL enable
	void	PWM_DCG_parser();	//PWM config
	void	PWM_DCG_frequency_parser();
	void	PWM_DCG_duty_cycle_parser();
	void	PWM_DCG_PWM_trig_parser();		//PWM triggering! not ext. triggering

	void	FWDG_parser();		//DCA Forward
	void	RFLG_parser();		//DCA Reflected
	void	CSG_parser();		//Clock source / Chain mode
	void	PATG_parser();		//PA type
	void	PCG_parser();		//Phase
	void	PWRG_parser();		//Power (W)
	void	PWRDG_parser();		//Power (dBm)
	void	VER_parser();		//Version
	void	GCG_parser(bool console_output_enabled = true);		//Get VGA attenuation
	void	MCG_parser(bool console_output_enabled = true);		//Get IQ Mod magnitude
	void	PWRSGDG_parser();	//SGx Board power output

	void	ECG_parser(bool console_output_enabled = true);		//RF enable
	void	FCG_parser(bool console_output_enabled = true);		//Frequency
	void	PTG_parser(bool console_output_enabled = true);		//Temperature
	void	ST_parser(bool console_output_enabled = true);		//Error status

	void	PSUEG_parser();		//PSU enable
	void	PSUVG_parser();		//PSU voltage setpoint
	void	PSUIG_parser();		//PSU current limit


	/* NUMPAD */
	/* target_parameter and target_button can be used to 'dynamically' refer to the most recently pressed 'control button' and accompanying relevant variable, on any of the pages.
	 * precision is used to decide how many decimals will be used.
	 * multiplier is for swapping between for example Hz and MHz.
	 * The contents of these four can be changed by using the numpad_preparation function below. */
	Numpad *numpad;
	QPushButton *target_parameter_button;
	int	 precision;
	double  scale_multiplier;

	void	numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision = 2, double scale = 1, bool period_enable = true, bool sign_enable = false);
	void	ok_check_minmax(QPushButton *button);		//Check if the value fulfill the min/max requirements of an input.

#ifdef ROTARY_ENCODER_ENABLED
	QThread* thread1 = new QThread;
	RotaryEncoderThread* RE_thread = new RotaryEncoderThread;
	QTimer *RE_update_timer;

	int		menu_index = 0;
	int		RE_freq_multiplier = 1;
	int		RE_pwr_multiplier = 1;
	bool	menu_enter = false;
	void	move_menu_index(int	i); // 0 = Frequency, 1 = power, 2 = RF button
	void	RE_menu_indicator(); // 0 = move around menu, 1 == item selected, 2 ==
	void	RCM_GUI_Updater();

signals:
	void	RE_threadSTOP();
#endif

};

#endif //MAINWINDOW_H
